//
//  AsReaderDeviceProtocol.h
//  AsReaderDockSDK
//
//  @brief  This file defines a protocol called AsReaderDeviceDelegate, which is used for managing hardware events related to AsReader devices.
//
//  Created by mac on 2017/9/7.
//  Copyright © 2017年 ZYL. All rights reserved.
//

#ifndef AsReaderDeviceProtocol_h
#define AsReaderDeviceProtocol_h
#endif /* AsReaderDeviceProtocol_h */

/**
 *  @brief      sioMgrDelegate
 *  @details    Delegate for hardware event callback.
 */
@protocol AsReaderDeviceDelegate <NSObject>
@required
/**
 *  @brief      Response to change in plug state
 *  @details    Function that is called when the plug state of the reader and iOS device changes
 *  @param      plug : plugged (YES), unplugged (NO)
 */
- (void)plugged:(BOOL)plug;


/**
 *  @brief      Notification about "Power Reset" from module.
 *  @details    Function that is called when the reader's connection status changes
 *  @param      status : Connected (OxFF), Disconnected(0x00)
 */
- (void)readerConnected:(int)status;

@optional
/**
 *  @brief      Response of trigger button on reader (Default)
 *  @details    This function is called when the trigger button of the reader is pressed or released.
 *  @param      strStatus : When device type is RFID, "RFID startScan"\n When device type is Barcode, "Barcode startScan"\n When device type is NFC, "NFC startScan"\n
 */
- (void)checkTriggerStatus:(NSString*)strStatus;


//20210928 Dual Mode Change Issue  by Doa
/**
 *  @brief      Response to "setReaderPower".
 *  @details    Function that is called when the reader sends a response code to "setReaderPower"
 *  @param      isOn : On (YES), Off(NO)
 *  @param      isHWModeChange : YES (When mode is changed by H/W switch)
 */
- (void)responsePowerOnOff:(BOOL)isOn HWModeChange:(BOOL)isHWModeChange;

/**
 *  @brief      [Customer Mode] Response to trigger button on reader
 *  @details    This function is called when the trigger button of the reader is released.
 */
- (void)releasedTriggerButton;

/**
 *  @brief      [Customer Mode] Response to trigger button on reader
 *  @details    This function is called when the trigger button of the reader is pressed.
 */
- (void)pushedTriggerButton;

/**
 *  @brief      Response to trigger button on reader (Default)
 *  @details    This function is called when the trigger button of the reader is pressed or released.
 *  @param      status : When device type is RFID, "RFID startScan"\n When device type is Barcode, "Barcode startScan"\n When device type is NFC, "NFC startScan"\n
 */
- (void)onAsReaderTriggerKeyEventStatus:(NSString*)status;

/**
 *  @brief      This function is called when a tag’s data is received
 *  @param      readData : Data of tag read
 */
- (void)receivedScanData:(NSData *)readData;

/**
*  @brief      This function is called when a tag’s data is received
*  @param      readData : Data of tag read
*  @param      dataType : Unknown(99), Barcode(0), RFID(1), NFC(2)
*/
- (void)receivedScanData:(NSData *)readData dataType:(int)dataType;

/**
 *  @brief      Response to undefined command
 *  @param      commandCode : Data (Payload)
 */
- (void)unknownCommandReceived:(int)commandCode;

/**
 *  @brief      Function that is called when a tag’s data is received (all type)
 *  @param      data : Data of tag read
 */
- (void)allDataReceived:(NSData *)data;

/**
 *  @brief      This function is called when the battery level of the reader is received
 *  @param      battery : Battery level
 */
- (void)batteryReceived:(int)battery;

/**
 *  @brief      Response to "StopAutoRead"
 *  @details    Function that is called when the reader sends a response code to "stopReadTags"
 *  @param      status : Success (0x00)
 */
- (void)stopReadScan:(int)status;

/**
 *  @brief      Response to "StartAutoRead"
 *  @details    Function that is called when the reader sends a response code to "startReadTags"
 *  @param      status : Success (0x00)
 */
- (void)startedReadScan:(int)status;

/**
 *  @brief      Response to invalid command
 *  @param      errorCode : payload(error code, command code, sub error code)
 */
- (void)errorReceived:(NSData *)errorCode;


- (void)receivedSleepTime:(int)time isAck:(BOOL)isAck;
- (void)receivedSleepTime:(int)time;

- (void)receivedBeep:(NSData *)data;

- (void)readerInfoReceived:(NSData *)data;

/**
 *  @brief      Response to "Set Stop Condition"
 *  @details    Function that is called when a response code to "Set Stop Condition" is received
 *  @param      status : Success (0x00)
 */
- (void)didSetStopCondition:(int)status;

/**
 *  @brief      Response to "Get Stop Condition"
 *  @details    Function that is called when a response code to "Get Stop Condition" is received
 *  @param      data : Tag count(1byte), Scan time(1byte), Cycle(1tyte)
 */
- (void)stopConditionsReceived:(NSData *)data;

/**
 *  @brief      [F/W Update] Function that is called when firmware update data is received
 *  @param      FWData : Success (0x00)
 */
- (void)responseBootloaderS0:(NSData *)FWData;
@end
