#import "InventoryViewController.h"
#import "InventoryTableViewCell.h"
#import "UserData.h"
#import "ReadMemoryViewController.h"
#import "WriteMemoryViewController.h"
#import "TagAccessViewController.h"
#import "SystemSetting.h"
#import "CSVData.h"
#import "DatabaseManeger.h"
#import "MBProgressHUD.h"
#import "CSVExportTableViewController.h"
#import "Define.h"
#import "ExcelData.h"
#import "RFMHelper.h"
#import "RainTagHelper.h"

#define ALERT_VIEW_OPERATION_TIME   1
#define ALERT_VIEW_TID_LEN          2
#define PICKER_VIEW_INVENTORY_SESSION   0
#define PICKER_VIEW_POWER_GAIN          1
#define PICKER_VIEW_SESSION_FLAG        2
#define csvInvenShow @"csvInventoryShow"

@interface InventoryViewController () {
    BOOL _isReloadNeeded;
    MaskType maskType;
    BOOL isCanScan;
    BOOL isJapan;
    BOOL isfinishedInitial;
    SystemSetting * systemSetting;
    UIAlertView * currentAlertView;
    DatabaseManeger * databaseManeger;
    
    NSInteger TPSCounter;
}

@property (strong, nonatomic) IBOutlet UITableView *tableView;
@property (weak, nonatomic) IBOutlet UIPickerView *pickerView;
@property (weak, nonatomic) IBOutlet UIButton *inventoryBtn;
@property (weak, nonatomic) IBOutlet UIButton *inventorySensorBtn;
@property (weak, nonatomic) IBOutlet UIBarButtonItem *csvOutputBtn;
@property (weak, nonatomic) IBOutlet UILabel *speedLabel;
@property (weak, nonatomic) IBOutlet UILabel *tagCountLabel;
@property (weak, nonatomic) IBOutlet UILabel *totalTagCountLabel;
@property (weak, nonatomic) IBOutlet UILabel *tidLabel;
@property (weak, nonatomic) IBOutlet UISwitch *displayPcSwitch;
@property (weak, nonatomic) IBOutlet UISwitch *continuousModeSwitch;
@property (weak, nonatomic) IBOutlet UISwitch *reportRssiSwitch;
@property (weak, nonatomic) IBOutlet UISwitch *reportTidSwitch;
@property (weak, nonatomic) IBOutlet UIView *powerGainPickerTopView;
@property (weak, nonatomic) IBOutlet UITextField *powerGainTextField;
@property (weak, nonatomic) IBOutlet UIScrollView *scrollView;
@property (weak, nonatomic) IBOutlet UITextField *operationTimeTextField;
@property (strong, nonatomic) IBOutletCollection(UIControl) NSArray *toDisableElems;
@property (strong, nonatomic) NSArray *powerGainPickerData;
@property (weak, nonatomic) IBOutlet UIButton *OptionButton;
@property (weak, nonatomic) IBOutlet UITextField *inventorySessionTextField;

@property (readwrite, nonatomic) SessionType inventorySession;
@property (strong, nonatomic) NSArray *inventorySessionPickerData;
@property (weak, nonatomic) IBOutlet UITextField *sessionFlagTextField;
@property (readwrite, nonatomic) SessionFlag sessionFlag;
@property (strong, nonatomic) NSArray *sessionFlagPickerData;
@property (strong, nonatomic) UIAlertController *actionAlertVC;
@property (assign, nonatomic)NSInteger csvType;

@end

@implementation InventoryViewController

- (void)viewDidLoad {
    [super viewDidLoad];
    systemSetting = [SystemSetting sharedSystemSetting];
    _isReloadNeeded = YES;
    
    appDelegate = (AppDelegate *)[[UIApplication sharedApplication] delegate];
    [appDelegate.mReader setDelegate:self];
    appDelegate.insertOperation.maxConcurrentOperationCount=1;
    
    TPSCounter = 0;
    
    databaseManeger = [DatabaseManeger sharedDatabaseManeger];
    
        
    [[databaseManeger getDatebaseQueue] inDatabase:^(FMDatabase *db) {
        mapTags = [[NSMutableDictionary alloc] init];
        aryTags = [[NSMutableArray alloc] initWithArray:[databaseManeger getAllAsTagItemsfmdb:db]];
        
        for (AsTagItem * item in  aryTags){
            if(item.tid.length == 0 ){
                if(item.celsius.length > 0 )
                    [mapTags setObject:item forKey:item.tag];
                else
                    [mapTags setObject:item forKey:[RainTagHelper convertUniqueStringDataTag:item.tag]];
            }
            else
                [mapTags setObject:item forKey:item.tag];
        }
    }];
    
    
    if([systemSetting getTidOn])
        [self.reportTidSwitch setOn:YES];
    else
        [self.reportTidSwitch setOn:NO];
    
    [AppDelegate showGlobalProgressHUDWithTitle:@"Loading"];
}

- (UIAlertController *)actionAlertVC {
    if (!_actionAlertVC) {
        _actionAlertVC = [UIAlertController alertControllerWithTitle:nil message:nil preferredStyle:UIAlertControllerStyleActionSheet];
        
        UIAlertAction *csvAction = [UIAlertAction actionWithTitle:@"CSV Export" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
            [self csvExport];
        }];
        
        UIAlertAction *csvRFMAction = [UIAlertAction actionWithTitle:@"Export CSV for RFM" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
            [self csvRFMExport];
        }];
        
        UIAlertAction *excelAction = [UIAlertAction actionWithTitle:@"Excel Export" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
            [self excelExport];
        }];
        UIAlertAction *excelListAction = [UIAlertAction actionWithTitle:@"Excel List" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
            
            [self replaceDB];
            [self.inventoryBtn setTitle:@"Inventory" forState:UIControlStateNormal];
            
            [[self toDisableElems]enumerateObjectsUsingBlock:^(id  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
                UIControl *control = obj;
                [control setEnabled:YES];
            }];
            
            self.csvOutputBtn.enabled = YES;
            self.tableView.userInteractionEnabled = YES;

            _csvType = 2;
            [self performSegueWithIdentifier:csvInvenShow sender:nil];
        }];
        UIAlertAction *cancelAction = [UIAlertAction actionWithTitle:@"Cancel" style:UIAlertActionStyleCancel handler:nil];

        [_actionAlertVC addAction:csvAction];
        [_actionAlertVC addAction:csvRFMAction];
        [_actionAlertVC addAction:excelAction];
        [_actionAlertVC addAction:excelListAction];
        [_actionAlertVC addAction:cancelAction];
    }
    return _actionAlertVC;
}

- (void)updateUI {
    [self.tagCountLabel setText:[NSString stringWithFormat:@"%d", (int)aryTags.count]];
    [self.totalTagCountLabel setText:[NSString stringWithFormat:@"%ld", mTotalTagCounts]];
    [self.speedLabel setText:[NSString stringWithFormat:@"%ld tps", TPSCounter]];
    [self.tableView reloadData];
}
- (void)viewWillAppear:(BOOL)animated {
    [super viewWillAppear:animated];
    [self setInventorySession:appDelegate.mReader.inventorySession];
    [self setSessionFlag:appDelegate.mReader.sessionFlag];
}

- (void)viewDidAppear:(BOOL)animated{
    isfinishedInitial = NO;
    NSNotificationCenter *center = [NSNotificationCenter defaultCenter];
    [center addObserver:self selector:@selector(AsReaderGUNDisconnected:) name:@"AsReaderGUNDisconnected" object:nil];
    [center addObserver:self selector:@selector(AsReaderGUNConnected:) name:@"AsReaderGUNConnected" object:nil];
    
    mTotalTagCounts = 0;
    for (AsTagItem * item in aryTags) {
        mTotalTagCounts  =  mTotalTagCounts +  item.count;
    }
   
    isJapan = [[appDelegate.mReader regionName] hasPrefix:@"Japan"];
    
    [self.speedLabel setText:@"0 tps"];
    [self.reportRssiSwitch setOn:NO];
    isCanScan = YES;
    
    [appDelegate.mReader setDelegate:self];
    
    
    if (_isReloadNeeded == NO) {
        isfinishedInitial = YES;
        return;
    }
    _isReloadNeeded = YES;
    
    maskType = appDelegate.mReader.maskTypeValue;
    appDelegate.mReader.storedMode = NO;
    
    appDelegate.mReader.reportMode = NO;
    
    [self initPowerGainPickerData];
    
    int nPower = appDelegate.mReader.powerGain;
    NSInteger operationTime = appDelegate.mReader.operationTime;

    [self setPowerGain:(int)(nPower / 10)];

    appDelegate.mReader.continuousMode = [systemSetting getContinuousMode];
    [self.continuousModeSwitch setOn:appDelegate.mReader.continuousMode];
    appDelegate.mReader.rssiMode = [systemSetting getRssiMode];
    [self.reportRssiSwitch setOn:appDelegate.mReader.rssiMode];
    [self.displayPcSwitch setOn:[systemSetting getDisPlayPC]];
    [self setOperationTime:[NSString stringWithFormat:@"%ld", (long)operationTime]];
    [self setInventorySession:appDelegate.mReader.inventorySession];
    [self setSessionFlag:appDelegate.mReader.sessionFlag];
    
    [self.tidLabel setText:[NSString stringWithFormat:@"Report TID : (len:%d word)",[systemSetting getTidLength]]];
    [self.reportTidSwitch setOn:[systemSetting getTidOn]];
    [self.pickerView reloadAllComponents];
    [self updateUI];
    isfinishedInitial = YES;
    [AppDelegate dismissGlobalHUD];
    
    
//    //タイマー
//    [NSTimer scheduledTimerWithTimeInterval:0.1
//         target:self
//                                   selector:@selector(onTimer:)
//         userInfo:nil
//         repeats:YES];
}

- (void)viewWillDisappear:(BOOL)animated {
    [super viewWillDisappear:animated];
    [[NSNotificationCenter defaultCenter] removeObserver:self name:@"AsReaderGUNDisconnected" object:nil];
    [[NSNotificationCenter defaultCenter] removeObserver:self name:@"AsReaderGUNConnected" object:nil];

    [appDelegate.mReader setDelegate:nil];
    [appDelegate.mReader stopSync];
    [appDelegate.mReader stop];
    [self enableControlByActionState];
   
}

- (void)didReceiveMemoryWarning {
    [super didReceiveMemoryWarning];
}

#pragma mark - CSV Export
- (void)csvExport {
    dispatch_async(dispatch_get_main_queue(), ^{
        if (aryTags && aryTags.count != 0) {
            NSString *timestr = [self currentData];
            NSString *fileName = [NSString stringWithFormat:@"RFID_%@.csv", timestr];
            //创建csv文件
            [CSVData createCSVWithFileName:fileName export:rfidExportType];
            BOOL isRssiOn = self.reportRssiSwitch.isOn;
            BOOL isTidOn = self.reportTidSwitch.isOn;
            [CSVData exportRfidCSV:fileName withArray:aryTags export:rfidExportType isEpc:self.displayPcSwitch.isOn isRssi:isRssiOn isTid:isTidOn];
        }
    });
}

- (void)csvRFMExport {
    dispatch_async(dispatch_get_main_queue(), ^{
        if (aryTags && aryTags.count != 0) {
            NSString *timestr = [self currentData];
            NSString *fileName = [NSString stringWithFormat:@"RFID_RFM_%@.csv", timestr];
            //创建csv文件
            [CSVData createCSVWithFileName:fileName export:rfidExportType];
            [CSVData exportRFMRfidCSV:fileName withArray:aryTags export:rfidExportType];
        }
    });
}


- (void)excelExport {
    dispatch_async(dispatch_get_main_queue(), ^{
        if (aryTags && aryTags.count != 0) {
            NSString *timestr = [self currentData];
            NSString *fileName = [NSString stringWithFormat:@"RFID_%@.xlsx", timestr];
            //创建excel文件
            [ExcelData createExcelWithFileName:fileName export:rfidExportType];
            BOOL isRssiOn = self.reportRssiSwitch.isOn;
            BOOL isTidOn = self.reportTidSwitch.isOn;
            [ExcelData exportRfidExcel:fileName withArray:aryTags export:rfidExportType isEpc:self.displayPcSwitch.isOn isRssi:isRssiOn isTid:isTidOn];
        }
    });
}

- (NSString *)currentData {
    NSDate *today = [NSDate date];
    NSDateFormatter *df = [[NSDateFormatter alloc] init];
    [df setDateFormat:@"yyyyMMddHHmmss"];
    NSString *str = [df stringFromDate:today];
    return str;
}

#pragma mark - Navigation
- (void)prepareForSegue:(UIStoryboardSegue *)segue sender:(id)sender {
    _isReloadNeeded = YES;
    
    if ([[segue identifier] isEqualToString:@"ReadMemory"]) {
        ReadMemoryViewController *destinationController = [segue destinationViewController];
        
        NSIndexPath *indexPath = (NSIndexPath *)sender;
        destinationController.maskValue = ((AsTagItem *)[aryTags objectAtIndex:indexPath.row]).tag;
    } else if ([[segue identifier] isEqualToString:@"WriteMemory"]) {
        WriteMemoryViewController *destinationController = [segue destinationViewController];
        
        NSIndexPath *indexPath = (NSIndexPath *)sender;
        destinationController.maskValue = ((AsTagItem *)[aryTags objectAtIndex:indexPath.row]).tag;
    } else if ([[segue identifier] isEqualToString:@"LockMemory"]) {
        TagAccessViewController *destinationController = [segue destinationViewController];
        
        NSIndexPath *indexPath = (NSIndexPath *)sender;
        destinationController.maskValue = ((AsTagItem *)[aryTags objectAtIndex:indexPath.row]).tag;
    } else if ([[segue identifier] isEqualToString:csvInvenShow]) {
        CSVExportTableViewController *exportVC = segue.destinationViewController;
        if (1 == _csvType) {
            exportVC.typeStr = @"csvInven";
        } else {
            exportVC.typeStr = @"excelInven";
        }
    }
}

#pragma mark - Storyboard Segue
-(IBAction)prepareForUnwind:(UIStoryboardSegue *)segue {
}

#pragma mark - UITableViewDataSource
- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView {
    return 1;
}

- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section {
    return aryTags.count;
}

#pragma mark - UITableViewDelegate
- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath {
    InventoryTableViewCell *cell = [tableView dequeueReusableCellWithIdentifier:@"InventoryCell" forIndexPath:indexPath];
    
    AsTagItem *item = nil;
    item = [aryTags objectAtIndex:indexPath.row];
    if (item != nil) {
        
        
        if (self.displayPcSwitch.isOn) {
            cell.pcLabel.text = [NSString stringWithFormat:@"%@ ",[RainTagHelper getPCStringData:item.tag]];
            cell.epcLabel.text = [NSString stringWithFormat:@"%@ ",[RainTagHelper getEPCStringData:item.tag]];
        } else {
            
            cell.pcLabel.text =@"";
            
            cell.epcLabel.text = [NSString stringWithFormat:@"%@",[RainTagHelper getEPCStringData:item.tag]];
        }
        
        
        cell.countLabel.text = [NSString stringWithFormat:@"%d", item.count];
        
        cell.tidLabel.text =[NSString stringWithFormat:@"%@", item.tid];
        
        
        if(item.celsius.length > 2){
            cell.rfmLabel.text = [NSString stringWithFormat:@"RFM : %@°F (%@°C) / M : %@", item.fahrenheit,item.celsius,item.moisture];
        }else{
            cell.rfmLabel.text = @"";
        }
        
        
        if (self.reportRssiSwitch.isOn) {
            cell.rssiLabel.text = [NSString stringWithFormat:@"%.1f dB", item.rssi];
            cell.phaseLabel.text = [NSString stringWithFormat:@"%d˚", (int)item.phase];
            cell.frequency.text =  [NSString stringWithFormat:@"%.2fMHz ",item.frequency];
        } else {
            cell.rssiLabel.text = @"";
            cell.phaseLabel.text = @"";
            cell.frequency.text = @"";
        }
    }
    return cell;
}

- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath {
    
    if ([appDelegate.mReader getAction] != CommandStop)
    {
        return;
    }
    
    UIAlertController *alert = [UIAlertController alertControllerWithTitle:@"Additional Actions"
                                                                   message:@"You can choose one of the following actions for this Tag"
                                                            preferredStyle:UIAlertControllerStyleAlert];
    
    
    [alert addAction:[UIAlertAction actionWithTitle:@"EPC Information"
                                              style:UIAlertActionStyleDefault
                                            handler:^(UIAlertAction * action) {
        
                                            AsTagItem *item = [aryTags objectAtIndex:indexPath.row];
                                            NSString *detailEPC =  [NSString stringWithFormat:@"%@",[RainTagHelper getEPCStringData:item.tag]];
                                            UIAlertView * aleart = [[UIAlertView alloc] initWithTitle:@"" message:detailEPC  delegate:nil cancelButtonTitle:@"OK" otherButtonTitles: nil];
                                            [aleart show];
        
                                            }]];
    
    
    [alert addAction:[UIAlertAction actionWithTitle:@"Read Memory"
                                              style:UIAlertActionStyleDefault
                                            handler:^(UIAlertAction * action) {
                                                [self nowViewWillDisappear];
                                                [self performSegueWithIdentifier:@"ReadMemory" sender:indexPath];
                                            }]];
    
    [alert addAction:[UIAlertAction actionWithTitle:@"Write Memory"
                                              style:UIAlertActionStyleDefault
                                            handler:^(UIAlertAction * action) {
                                                [self nowViewWillDisappear];
                                                [self performSegueWithIdentifier:@"WriteMemory" sender:indexPath];
                                            }]];
    
    [alert addAction:[UIAlertAction actionWithTitle:@"Lock Memory"
                                              style:UIAlertActionStyleDefault
                                            handler:^(UIAlertAction * action) {
                                                [self nowViewWillDisappear];
                                                [self performSegueWithIdentifier:@"LockMemory" sender:indexPath];
                                            }]];
    
    [alert addAction:[UIAlertAction actionWithTitle:@"Check Tag State(Lock)"
                                              style:UIAlertActionStyleDefault
                                            handler:^(UIAlertAction * action) {
        
        NSString *pcEpc = ((AsTagItem *)[aryTags objectAtIndex:indexPath.row]).tag;
        TagState state =  [appDelegate.mReader getTagState:pcEpc];
        NSString *result = @"";
        switch (state) {
			case State_Unknown:
                result = @"Unknown";
                break;
			case State_Error:
                result = @"Error";
                break;
			case State_NoReply:
                result = @"NoReply";
                break;				
			case State_UnLock:
                result = @"UnLock";
                break;				
            case State_Lock:
                result = @"Lock";
                break;
            case State_PermaLock:
				result = @"PermaLock";
				break;
            default:
                break;
        }
        
        UIAlertView * alert = [[UIAlertView alloc] initWithTitle:@"" message:result delegate:self cancelButtonTitle:@"OK" otherButtonTitles: nil];
        [alert show];
    }]];
    
        
    [alert addAction:[UIAlertAction actionWithTitle:@"Cancel"
                                              style:UIAlertActionStyleCancel
                                            handler:^(UIAlertAction *action) {
                                                
                                            }]];
    
    [self presentViewController:alert animated:YES completion:nil];
}

#pragma mark - UIPickerViewDataSource
- (NSInteger)numberOfComponentsInPickerView:(UIPickerView *)pickerView
{
    return 1;
}
- (NSInteger)pickerView:(UIPickerView *)pickerView numberOfRowsInComponent:(NSInteger)component
{
    if (pickerView.tag == PICKER_VIEW_POWER_GAIN) {
        return [self.powerGainPickerData count];
    } else if (pickerView.tag == PICKER_VIEW_INVENTORY_SESSION) {
        return self.inventorySessionPickerData.count;
    } else if (pickerView.tag == PICKER_VIEW_SESSION_FLAG) {
        return self.sessionFlagPickerData.count;
    }
    return 0;
}

#pragma mark - UIPickerViewDelegate
- (NSString *)pickerView:(UIPickerView *)pickerView titleForRow:(NSInteger)row forComponent:(NSInteger)component
{
    if (pickerView.tag == PICKER_VIEW_POWER_GAIN) {
        return [self.powerGainPickerData objectAtIndex:row];
    } else if (pickerView.tag == PICKER_VIEW_INVENTORY_SESSION) {
        return [self.inventorySessionPickerData objectAtIndex:row];
    } else if (pickerView.tag == PICKER_VIEW_SESSION_FLAG) {
        return [self.sessionFlagPickerData objectAtIndex:row];
    }
    return @"None";
}

#pragma mark - UITextViewDelegate
- (BOOL)textFieldShouldBeginEditing:(UITextField *)textField
{
    return false;
}

#pragma mark - UIAlertViewDelegate
- (void)alertView:(UIAlertView *)alertView clickedButtonAtIndex:(NSInteger)buttonIndex
{
    if (alertView.tag == ALERT_VIEW_OPERATION_TIME) {
        if (buttonIndex == 1) {
            NSString * value = [alertView textFieldAtIndex:0].text;
            if(value.length == 0){
               value = @"0";
            }

            [self setOperationTime:value];
            appDelegate.mReader.operationTime = [value intValue];
            [systemSetting setOperationTime:[value intValue]];
        }
    }
    
    if (alertView.tag == ALERT_VIEW_TID_LEN) {
        if (buttonIndex == 1) {
            int len = 1;
            NSString * value = [alertView textFieldAtIndex:0].text;
            if([value intValue] < 1){
               len = 1;
            }
            else len = [value intValue];
            [systemSetting setTidLength:len];
            NSString *strTid = [NSString stringWithFormat:@"Report TID (len : %d word)",len];
            [self.tidLabel setText:strTid];
        }
    }
    
}

#pragma mark - IBActions
- (IBAction)powerGainBtnTapped:(id)sender {
    self.pickerView.tag = PICKER_VIEW_POWER_GAIN;
    [self.pickerView reloadAllComponents];
    if (![self.powerGainTextField.text isEqualToString:@""]) {
        [self.pickerView selectRow:[self.powerGainPickerData indexOfObject:self.powerGainTextField.text] inComponent:0 animated:NO];
    } else {
        [self.pickerView selectRow:0 inComponent:0 animated:NO];
    }
    self.pickerView.hidden = NO;
    self.powerGainPickerTopView.hidden = NO;
}

- (IBAction)displayPcSwitchTapped:(id)sender {
    [systemSetting setDisPlayPC:[self.displayPcSwitch isOn]];
    [self.tableView reloadData];
}

- (IBAction)continuousModeSwitchTapped:(id)sender {
    [appDelegate.mReader stop];
    appDelegate.mReader.continuousMode = [self.continuousModeSwitch isOn];
    [systemSetting setContinuousMode:[self.continuousModeSwitch isOn]];
}

- (IBAction)displayTidSwitchTapped:(id)sender {
    
    [systemSetting setTidOn:_reportTidSwitch.isOn];
    
    if(_reportTidSwitch.isOn)
    {
        UIAlertView *alert = [[UIAlertView alloc] initWithTitle:@"TID Length:" message:@"Please enter TID length(word).\n(ex:1 word = 2byte )" delegate:self cancelButtonTitle:@"Cancel" otherButtonTitles:nil];
        
        alert.alertViewStyle = UIAlertViewStylePlainTextInput;
        [alert textFieldAtIndex:0].keyboardType = UIKeyboardTypeNumberPad;
        [alert textFieldAtIndex:0].text = [NSString stringWithFormat:@"%d",[systemSetting getTidLength]];
        alert.tag = ALERT_VIEW_TID_LEN;
        
        [alert addButtonWithTitle:@"OK"];
        [alert show];
    }
    //[self.tableView reloadData];
}


- (IBAction)reportRssiSwitchTapped:(id)sender {
    appDelegate.mReader.rssiMode = [self.reportRssiSwitch isOn];
    [systemSetting setRssiMode:[self.reportRssiSwitch isOn]];
//    [aryTags removeAllObjects];
//    [mapTags removeAllObjects];
//    [appDelegate.insertOperation cancelAllOperations];
//    [[databaseManeger getDatebaseQueue] inTransaction:^(FMDatabase *db, BOOL *rollback) {
//        if (![databaseManeger deleteAllAsTagItemsFmdb:db]) {
//            *rollback = YES;
//        }
//    }];
//    currentTagCount = 0;
//    [self.tagCountLabel setText:[NSString stringWithFormat:@"%d", (int)aryTags.count]];
//    [self.totalTagCountLabel setText:[NSString stringWithFormat:@"%d", (int)aryTags.count]];
//    [self.speedLabel setText:@"0 tps"];
//    mTagSpeed = 0.0;
//    mLastTime = 0;
//    mTotalTagCount = 0;
//    mTotalTagCounts = 0;
    [self.tableView reloadData];
}

- (IBAction)inventoryBtnTapped:(id)sender {
    
    if ([appDelegate.mReader getAction] == CommandStop) {
         if(_reportTidSwitch.isOn) {
             [appDelegate.mReader inventoryWithTidOffset:0 length:[systemSetting getTidLength] inventorySession:_inventorySession sessionFlag:_sessionFlag];
         }
         else
             [appDelegate.mReader inventory];
    } else {
        [appDelegate.mReader stop];
    }
}

- (IBAction)clearBtnTapped:(id)sender {
    [aryTags removeAllObjects];
    [mapTags removeAllObjects];
    [self.tagCountLabel setText:[NSString stringWithFormat:@"%d", (int)aryTags.count]];
    [self.totalTagCountLabel setText:[NSString stringWithFormat:@"%d", (int)aryTags.count]];
    [self.speedLabel setText:@"0 tps"];
    TPSCounter = 0;
    mTotalTagCounts = 0;
    [appDelegate.insertOperation cancelAllOperations];
    [[databaseManeger getDatebaseQueue] inTransaction:^(FMDatabase *db, BOOL *rollback) {
        if (![databaseManeger deleteAllAsTagItemsFmdb:db]) {
            *rollback = YES;
        }
    }];
    [self.tableView reloadData];
}


- (IBAction)maskBtnTapped:(id)sender {
    if (maskType == MaskType_NO_MASK) {
        UIAlertView * alert = [[UIAlertView alloc] initWithTitle:@"" message:@"Please set the mask type in RFID Options menu before using mask." delegate:self cancelButtonTitle:@"OK" otherButtonTitles: nil];
        [alert show];
        return;
    }else if (maskType == MaskType_Selection) {
        [self nowViewWillDisappear];
        [self performSegueWithIdentifier:@"SelectionMask" sender:self];
    } else if (maskType == MaskType_EPC) {
        [self nowViewWillDisappear];
        [self performSegueWithIdentifier:@"EpcMask" sender:self];
    }
    _isReloadNeeded = NO;
}

- (IBAction)pickerCloseBtnTapped:(id)sender {
    [self hidePickerView];
}

- (IBAction)powerGainPickerOkBtnTapped:(id)sender {
    if (self.pickerView.tag == PICKER_VIEW_POWER_GAIN) {
        NSInteger idx = [self.pickerView selectedRowInComponent:0];
        idx += (appDelegate.mReader.powerGainRange.min / 10);
        [self setPowerGain:idx];
        appDelegate.mReader.powerGain = (int)idx * 10;
        [systemSetting setPowerGain:(int)idx * 10];
    }else if(self.pickerView.tag == PICKER_VIEW_INVENTORY_SESSION){
        [self setInventorySession:(int)[self.pickerView selectedRowInComponent:0]];
        [appDelegate.mReader setInventorySession:self.inventorySession];
        [systemSetting setInventorySession:self.inventorySession];
    }else if(self.pickerView.tag == PICKER_VIEW_SESSION_FLAG){
        [self setSessionFlag:(int)[self.pickerView selectedRowInComponent:0]];
        [appDelegate.mReader setSessionFlag:self.sessionFlag];
        [systemSetting setSessionFlag:self.sessionFlag];
    }
    [self hidePickerView];
    
}

- (IBAction)operationTimeBtnTapped:(id)sender {
    UIAlertView *alert = [[UIAlertView alloc] initWithTitle:@"Operation Time:" message:@"Please enter operation time.\n(ex: 30 ms)" delegate:self cancelButtonTitle:@"Cancel" otherButtonTitles:nil];
    
    alert.alertViewStyle = UIAlertViewStylePlainTextInput;
    [alert textFieldAtIndex:0].keyboardType = UIKeyboardTypeNumberPad;
    alert.tag = ALERT_VIEW_OPERATION_TIME;
    
    [alert addButtonWithTitle:@"OK"];
    [alert show];
}

- (IBAction)csvOutputBtnTapped:(id)sender {
    [self presentViewController:self.actionAlertVC animated:NO completion:nil];

}

- (IBAction)backBtnTapped:(id)sender {
    [self nowViewWillDisappear];
    [self.navigationController popViewControllerAnimated:YES];
}


#pragma mark - Getters
- (NSArray *)inventorySessionPickerData {
    return DATA_INVENTORY_SESSION;
}
- (NSArray *)sessionFlagPickerData {
    return DATA_SESSION_FLAG;
}

#pragma mark - Init
- (void)initPowerGainPickerData {
    if (_powerGainPickerData == nil) {
        NSMutableArray *tmpArray = [[NSMutableArray alloc] init];
        int min = appDelegate.mReader.powerGainRange.min;
        int max = appDelegate.mReader.powerGainRange.max;
        
        for (int i = min; i <= max; i += 10) {
            [tmpArray addObject:[NSString stringWithFormat:@"%i.0 dB", i / 10]];
        }
        _powerGainPickerData = tmpArray;
    }
}

#pragma mark - Custom
- (void) hidePickerView {
    self.powerGainPickerTopView.hidden = YES;
    self.pickerView.hidden = YES;
}

- (void)setInventorySession:(SessionType)inventorySession {
    self.inventorySessionTextField.text = [self.inventorySessionPickerData objectAtIndex:inventorySession];
    _inventorySession = inventorySession;
}

- (void)setPowerGain:(NSInteger)power {
    NSString *value = [NSString stringWithFormat:@"%li.0 dB", (long)power];
    NSInteger index = [self.powerGainPickerData indexOfObject:value];
    if ([self.powerGainPickerData count] >index) {
        self.powerGainTextField.text = [self.powerGainPickerData objectAtIndex:index];
    }
}
- (void)setSessionFlag:(SessionFlag)sessionFlag {
    self.sessionFlagTextField.text = [self.sessionFlagPickerData objectAtIndex:sessionFlag];
    _sessionFlag = sessionFlag;
}

- (void)setOperationTime:(NSString *)operationTime {
    self.operationTimeTextField.text = operationTime;
}
- (void)changStatus{
    isCanScan = YES;
}
#pragma mark - Reader Event Handler Methods
- (BOOL)onAsReaderTriggerKeyEvent:(BOOL)status{
    if(!isfinishedInitial){
        NSLog(@"[Event] ============> isfinishedInitial" );
        return  NO;
    }
        
    
    if ([appDelegate.mReader getScanMode] == RFIDScanMode) {
        if (status) {
            if (isCanScan) {
                if(_reportTidSwitch.isOn)
                {
                   NSLog(@"[Event] ===>Inventory  TID START ");
                   [appDelegate.mReader inventoryWithTidOffset:0 length:[systemSetting getTidLength] inventorySession:_inventorySession sessionFlag:_sessionFlag];
                }
                else
                  [appDelegate.mReader inventory];
            }
        }else{
             [appDelegate.mReader stop];
        }
        return NO;
    }else{
        return YES;
    }
}
- (BOOL)onAsReaderLeftModeKeyEvent:(BOOL)status{
    if(appDelegate.mReader.isRFIDModule&&appDelegate.mReader.isBarcodeModule){
        return YES;
    }
    return NO;
}
- (BOOL)onAsReaderRightModeKeyEvent:(BOOL)status{
    if(appDelegate.mReader.isRFIDModule&&appDelegate.mReader.isBarcodeModule){
        return YES;
    }
    return NO;
}
- (void)updateDeviceState:(ResultCode)error {
    NSLog(@"ErrorCode = %04x",error );
    //    if (error != ResultNoError) {
    //        UIAlertView *alert = [[UIAlertView alloc]initWithTitle:@"Error" message:@"Failed to get response. Try again." delegate:nil cancelButtonTitle:@"OK" otherButtonTitles:nil];
    //        [alert show];
    //        [self.navigationController popViewControllerAnimated:YES];
    //    }
}

- (void)readTag:(NSString *)tag rssi:(float)rssi phase:(float)phase frequency:(float)frequency {
    if(_reportTidSwitch.isOn)
        return;
    
    __block float rssiBlock = rssi;
    __block float phaseBlock = phase;
    __block float frequencyBlock = frequency;
    dispatch_queue_t queue =  dispatch_queue_create("queue",DISPATCH_QUEUE_SERIAL);
    dispatch_async(queue, ^{
        
        mTotalTagCounts++;
        TPSCounter++;
        
        AsTagItem *item = nil;
        @synchronized (self) {
            //if ((item = (AsTagItem *)[mapTags objectForKey:tag]) != nil) {
            NSString *newData =[RainTagHelper convertUniqueStringDataTag:tag];
            
           
            
            if ((item = (AsTagItem *)[mapTags objectForKey:newData]) != nil) {
                item.count++;
                [item updateRssi:rssiBlock withPhase:phaseBlock frequency:frequencyBlock];
            } else {
                item = [AsTagItem itemWith:tag rssi:rssiBlock phase:phaseBlock frequency:frequencyBlock tid:@""];
                [aryTags addObject:item];
                [mapTags setObject:item forKey:newData];
            }
            
            dispatch_async(dispatch_get_main_queue(), ^{
                [self.tagCountLabel setText:[NSString stringWithFormat:@"%d", (int)aryTags.count]];
                [self.totalTagCountLabel setText:[NSString stringWithFormat:@"%ld", mTotalTagCounts]];
                
                if( [self calcTPS]>=1 ) {
                    [self.speedLabel setText:[NSString stringWithFormat:@"%ld tps", TPSCounter]];
                    TPSCounter = 0;
                }

                [self.tableView reloadData];
            });
        }
    });
}

- (void)onModuleOverHeated{
    [currentAlertView dismissWithClickedButtonIndex:0 animated:YES];
    currentAlertView = [[UIAlertView alloc] initWithTitle:@"" message:@"The RFID module is overheating. Please pause reading until it cools off." delegate:self cancelButtonTitle:@"OK" otherButtonTitles: nil];
    [currentAlertView show];
}
- (void)onResetReader{
    [currentAlertView dismissWithClickedButtonIndex:0 animated:YES];
    currentAlertView = [[UIAlertView alloc] initWithTitle:@"" message:@"Device must be rebooted due to packet loss." delegate:self cancelButtonTitle:@"OK" otherButtonTitles: nil];
    [currentAlertView show];
}

- (void)changedActionState:(CommandType)action resultCode:(NSInteger)resultCode {
    
    //NSLog(@"Action : %02X", action);
    
    if (action == CommandStop){
        
        TPSCounter = 0;
        
        [[self toDisableElems]enumerateObjectsUsingBlock:^(id  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
            UIControl *control = obj;
            [control setEnabled:NO];
        }];
        
        [_reportTidSwitch setEnabled:YES];
        
        [self replaceDB];
        [NSThread sleepForTimeInterval:0.3];
        [self setTitle:@"Inventory"];
        [self.inventoryBtn setTitle:@"Inventory" forState:UIControlStateNormal];

        [[self toDisableElems]enumerateObjectsUsingBlock:^(id  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
            UIControl *control = obj;
            [control setEnabled:YES];
        }];
        self.csvOutputBtn.enabled = YES;
        self.tableView.userInteractionEnabled = YES;
        if (resultCode == ResultNoError) {
            isCanScan = YES;
            [NSObject cancelPreviousPerformRequestsWithTarget:self selector:@selector(changStatus) object:nil];
            
        }else{
            isCanScan = NO;
            [NSObject cancelPreviousPerformRequestsWithTarget:self selector:@selector(changStatus) object:nil];
            [self performSelector:@selector(changStatus) withObject:nil afterDelay:1.0];
        }
    }else{
        
        if(action == CommandInventory){
            
            [[self toDisableElems]enumerateObjectsUsingBlock:^(id  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
                UIControl *control = obj;
                [control setEnabled:YES];
            }];
            
            [_reportTidSwitch setEnabled:NO];
            
        }
        
        
        if (resultCode == ResultNoError){
            [self.inventoryBtn setTitle:@"Stop" forState:UIControlStateNormal];
            [[self toDisableElems]enumerateObjectsUsingBlock:^(id  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
                UIControl *control = obj;
                [control setEnabled:NO];
                
            }];
            self.csvOutputBtn.enabled = YES;
            self.tableView.userInteractionEnabled = NO;
        }
    }
}

- (void)enableControlByActionState {
    if ([appDelegate.mReader getAction] == CommandStop){
        [self replaceDB];
        [self.inventoryBtn setTitle:@"Inventory" forState:UIControlStateNormal];
  
        TPSCounter = 0;
        
        [[self toDisableElems]enumerateObjectsUsingBlock:^(id  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
            UIControl *control = obj;
            [control setEnabled:YES];
        }];
        self.csvOutputBtn.enabled = YES;
        self.tableView.userInteractionEnabled = YES;
    }
}

- (void)readerInitialized:(AsReader *)reader {
    if (appDelegate.mReader.isBarcodeModule) {
        [AppDelegate showGlobalProgressHUDWithTitle:@"Barcode Setting..."];
        [CATransaction commit];
        [[SystemSetting sharedSystemSetting] barcodeSetting];
        [AppDelegate dismissGlobalHUD];
        [CATransaction commit];
    }
    if (appDelegate.mReader.isRFIDModule) {
        [AppDelegate showGlobalProgressHUDWithTitle:@"RFID Setting..."];
        [CATransaction commit];
        [[SystemSetting sharedSystemSetting] RFIDSetting];
        [AppDelegate dismissGlobalHUD];
        [CATransaction commit];
    }
}


- (IBAction)onClickOptionButton:(id)sender {
    
    if (self.scrollView.isHidden)
        self.scrollView.hidden = NO;
    else
        self.scrollView.hidden = YES;
}


- (IBAction)onClickInventorySessionButton:(id)sender {
    self.pickerView.tag = PICKER_VIEW_INVENTORY_SESSION;
    [self.pickerView reloadAllComponents];
    
    [self.pickerView selectRow:self.inventorySession inComponent:0 animated:NO];
    
    self.pickerView.hidden = NO;
    self.powerGainPickerTopView.hidden = NO;
}
- (IBAction)onClickSessionFlagButton:(id)sender {
    self.pickerView.tag = PICKER_VIEW_SESSION_FLAG;
    [self.pickerView reloadAllComponents];
    
    [self.pickerView selectRow:self.sessionFlag inComponent:0 animated:NO];
    
    self.pickerView.hidden = NO;
    self.powerGainPickerTopView.hidden = NO;
}

#pragma mark - NSNotification
- (void)AsReaderGUNDisconnected:(NSNotification *)notification {
    dispatch_async(dispatch_get_main_queue(), ^{
        appDelegate.mReader = nil;
        [self nowViewWillDisappear];
        [self.navigationController popToRootViewControllerAnimated:YES];
    });
}
- (void)AsReaderGUNConnected:(NSNotification *)notification {
    dispatch_async(dispatch_get_main_queue(), ^{
        [AppDelegate showGlobalProgressHUDWithTitle:@"Loading..."];
        [CATransaction commit];
        appDelegate.mReader = [[AsReader alloc] initWithAsReaderGUN:appDelegate.mAsReaderGUN delegate:self];
        [AppDelegate dismissGlobalHUD];
        [CATransaction commit];
    });
}

- (void)nowViewWillDisappear{
    [self replaceDB];
    if (appDelegate.insertOperation.operationCount > 0) {
        [AppDelegate dismissGlobalHUD];
        [AppDelegate showGlobalProgressHUDWithTitle:@"Data Insert..."];
        [CATransaction commit];
        NSLog(@"wait");
        while (true) {
            [NSThread sleepForTimeInterval:0.5];
            if (appDelegate.insertOperation.operationCount==0) {
                NSLog(@"wait");
                break;
            }
        }
        [AppDelegate dismissGlobalHUD];
        [CATransaction commit];
    }else{
        return;
    }
}

- (void)replaceDB{
    [appDelegate.insertOperation addOperationWithBlock:^{
        [[databaseManeger getDatebaseQueue] inTransaction:^(FMDatabase *db, BOOL *rollback) {
            for (int index =0; index<(int)[aryTags count]; index++)
            {
                AsTagItem *item = [aryTags objectAtIndex:index];
                
                NSString * ids = item.tag;
                
                if((item.tid.length > 0 )&&(item.celsius.length == 0))
                    ids = [NSString stringWithFormat:@"T%@",item.tag];
                    
                NSString * sql = [NSString stringWithFormat:@"REPLACE INTO AsTagItem \
                                  (ids\
                                  ,tag\
                                  ,count\
                                  ,rssi\
                                  ,phase\
                                  ,frequency\
                                  ,datetime\
                                  ,tid\
                                  ,temp1\
                                  ,temp2\
                                  ,moisture) \
                                  VALUES ('%@','%@','%@','%@','%@','%@','%@','%@','%@','%@','%@')",
                                  ids,
                                  item.tag,
                                  [NSString stringWithFormat:@"%d",item.count],
                                  [NSString stringWithFormat:@"%f",item.rssi],
                                  [NSString stringWithFormat:@"%f",item.phase],
                                  [NSString stringWithFormat:@"%f",item.frequency],
                                  item.dateTime,
                                  item.tid,
                                  item.celsius,
                                  item.fahrenheit,
                                  item.moisture];
                if (![databaseManeger updataOrAddSql:sql fmdb:db]) {
                    *rollback = YES;
                }
            }
        }];
    }];
}

- (void)readTagWithTid:(ResultCode)error actionState:(CommandType)action epc:(NSString *)epc tid:(NSString *)data rssi:(float)rssi phase:(float)phase frequency:(float)frequency {
    if (error != ResultNoError) {
        return;
    }

    __block float rssiBlock = rssi;
    __block float phaseBlock = phase;
    __block float frequencyBlock = frequency;
    dispatch_queue_t queue =  dispatch_queue_create("queue",DISPATCH_QUEUE_SERIAL);
    dispatch_async(queue, ^{

        TPSCounter++;
        mTotalTagCounts++;
        

        //NSString *newData =[RainTagHelper convertUniqueStringDataTag:epc];
        AsTagItem *item = nil;
        @synchronized (self) {
            if ((item = (AsTagItem *)[mapTags objectForKey:epc]) != nil) {
                item.count++;
                item.tid = data;
                [item updateRssi:rssiBlock withPhase:phaseBlock frequency:frequencyBlock];
            } else {
                item = [AsTagItem itemWith:epc rssi:rssiBlock phase:phaseBlock frequency:frequencyBlock tid:data];
                [aryTags addObject:item];
                [mapTags setObject:item forKey:epc];
            }
            dispatch_async(dispatch_get_main_queue(), ^{
                [self.tagCountLabel setText:[NSString stringWithFormat:@"%d", (int)aryTags.count]];
                [self.totalTagCountLabel setText:[NSString stringWithFormat:@"%ld", mTotalTagCounts]];
                
                if( [self calcTPS]>=1 ) {
                    [self.speedLabel setText:[NSString stringWithFormat:@"%ld tps", TPSCounter]];
                    TPSCounter = 0;
                }
                
                [self.tableView reloadData];
            });
        }
    });
}


//定期実行させる処理
- (void)onTimer:(NSTimer *)timer {
    [self calcTPS];
}

NSDate *startCalcTime = nil;
- (NSInteger)calcTPS {
    
    if( startCalcTime==nil ) {
        startCalcTime = [NSDate date];
    }
    
    NSDate *currentTime = [NSDate date];
    NSTimeInterval passedTime = [currentTime timeIntervalSinceDate:startCalcTime];
    
    //NSLog(@"--------- Time Test : Passed Time = %ld", (NSInteger)passedTime);
    
    if( passedTime>=1 ) {
        startCalcTime = nil;
    }
    return (NSInteger)passedTime;
}

- (void)readTagWithRFM:(ResultCode)error actionState:(CommandType)action epc:(NSString *)epc rssi:(float)rssi phase:(float)phase frequency:(float)frequency tid:(NSString *)tid temCode:(NSString*)tempCode calibaration:(NSString*)calibration sensorCode:(NSString*)sensorCode{
    if (error != ResultNoError) {
        return;
    }
    
    NSString *temp1 =[RFMHelper calculationTemp:tempCode calibration:calibration];
    NSString *temp2 =[RFMHelper convertCelsiusToFahrenheit:temp1];
    NSString *moisture =[RFMHelper calculationSensorCode:sensorCode tid:tid];
    

    
    dispatch_queue_t queue =  dispatch_queue_create("queue",DISPATCH_QUEUE_SERIAL);
    dispatch_async(queue, ^{

        TPSCounter++;
        mTotalTagCounts++;
               
        AsTagItem *item = nil;
        @synchronized (self) {
            if ((item = (AsTagItem *)[mapTags objectForKey:epc]) != nil) {
                item.count++;
                [item updateRFMTemp1:temp1 temp2:temp2 moisture:moisture];
            } else {
                item = [AsTagItem itemWithRFM:epc temp1:temp1 temp2:temp2 moisture:moisture];
                [aryTags addObject:item];
                [mapTags setObject:item forKey:epc];
            }
            dispatch_async(dispatch_get_main_queue(), ^{
                [self.tagCountLabel setText:[NSString stringWithFormat:@"%d", (int)aryTags.count]];
                [self.totalTagCountLabel setText:[NSString stringWithFormat:@"%ld", mTotalTagCounts]];
                
                if( [self calcTPS]>=1 ) {
                    [self.speedLabel setText:[NSString stringWithFormat:@"%ld tps", TPSCounter]];
                    TPSCounter = 0;
                }
                
                [self.tableView reloadData];
            });
        }
    });
}

-(IBAction)rfmSensorBtnTap:(id)sender{
    
    if (maskType != MaskType_NO_MASK) {
        UIAlertView * alert = [[UIAlertView alloc] initWithTitle:@"Please select NO MASK " message:@" (RFID Option -> MASK Type : No Mask)" delegate:self cancelButtonTitle:@"OK" otherButtonTitles: nil];
        [alert show];
        return;
    }
    
    [self setTitle:@"RFM Sensor (Beta)"];
    [appDelegate.mReader readRFMSensorTag];
}
@end
