//
//  BarcodeDemoViewController.swift
//  AsReaderGUN
//
//  Created by mac on 2018/7/11.
//  Copyright © 2018年 asterisk. All rights reserved.
//

import UIKit

final class BarcodeDemoViewController: UIViewController {
    @IBOutlet private weak var tableView: UITableView!
    @IBOutlet private weak var outputBtn: UIBarButtonItem!
    @IBOutlet private weak var pickerView: UIPickerView!
    @IBOutlet private weak var rescanTimePickerTopView: UIView!
    @IBOutlet private weak var rescanTimeTextField: UITextField!
    @IBOutlet private weak var startScanBtn: UIButton!
    @IBOutlet private weak var hwTriggerBtn: UIButton!
    @IBOutlet private weak var totalCountLabel: UILabel!
    @IBOutlet private var toDisableElems: [UIControl]!
    @IBOutlet var encodingSegment: UISegmentedControl!
    private var encoding: String.Encoding = String.Encoding.ascii
    private var asReaderGUNManager: AsReaderGUNManager! = AsReaderGUNManager.sharedAsReaderGUNManager
    private var systemSetting: SystemSetting! = SystemSetting.shared()
    private var isScanTimerEnable: Bool = false
    private var rescanTimeValue: Int = 0
    private var barcodeData: [AsBarcodeItem] = []
    private var exportType: Int = 0
    private lazy var rescanTimePickerData: [String] = {
        return rescanTimePickerArr
    }()
    private lazy var rescanTimeData: [String] = {
        return rescanTimeArr
    }()
    override func viewDidLoad() {
        super.viewDidLoad()
        encodingSegment.selectedSegmentIndex = 0
        rescanTimeTextField.text = rescanTimePickerData[Int(systemSetting.getBarcodeReScanTime())]
        rescanTimeValue = Int(systemSetting.getBarcodeReScanTime())
        if let barcodeArr =  systemSetting.getBarcodeData() {
            barcodeData = barcodeArr
        } else {
            barcodeData = []
        }
        totalCountLabel.text = "\(barcodeData.count)"
        AsReaderGUNManager.showGlobalProgressHUD(withTitle: loading)
        asReaderGUNManager = AsReaderGUNManager.sharedAsReaderGUNManager
        asReaderGUNManager.asReaderGUNManagerDelegate = self
    }
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
    }
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        AsReaderGUNManager.dismissGlobalHUD()
        asReaderGUNManager.mReader?.setBarcodeMode(true, isKeyAction: true, isCustomPreSuffixOn: systemSetting.getCustomPrefixSuffix(), type:Temporary)
        asReaderGUNManager.mReader?.setScanMode(BarcodeScanMode)
    }
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        if (asReaderGUNManager.mReader != nil) {
            asReaderGUNManager.mReader?.setScanMode(RFIDScanMode)
            asReaderGUNManager.mReader?.setBarcodeMode(false, isKeyAction: false, isCustomPreSuffixOn: systemSetting.getCustomPrefixSuffix(), type:Temporary)
        }
    }
    override func didReceiveMemoryWarning() {
        super.didReceiveMemoryWarning()
    }
    override func prepare(for segue: UIStoryboardSegue, sender: Any?) {
        if segue.identifier == "barcodeExport" {
            let exportVC = segue.destination as? ExportTableViewController
            if exportType == 1 {
                exportVC?.exportType = "csvBarcode"
            } else {
                exportVC?.exportType = "excelBarcode"
            }
        }
    }
    @IBAction func whenEncodingSegmentChanged(_ sender: Any) {
        switch (encodingSegment.selectedSegmentIndex) {
        case 0:
            encoding = String.Encoding.ascii;
            break;
        case 1:
            encoding = String.Encoding.utf8;
            break;
        case 2:
            encoding = String.Encoding.shiftJIS;
            break;
        default:
            encoding = String.Encoding.ascii;
            break;
        }
    }
    @IBAction private func clearBtnTapped(_ sender: UIButton) {
        barcodeData.removeAll()
        SystemSetting.shared()?.setBarcodeData(barcodeData)
        tableView.reloadData()
        totalCountLabel.text = "\(barcodeData.count)"
    }
    @IBAction private func scanStartBtnTapped(_ sender: UIButton) {
        setScanTimerEnable(true)
        scanEnabled(true)
    }
    @IBAction func scanStopBtnTapped(_ sender: UIButton) {
        stopScan()
    }
    @IBAction private func scanHWTriggerStartBtnTapped(_ sender: UIButton) {
        setScanTimerEnable(false)
        scanEnabled(false)
        let btn = sender
        let btnTitle: String = btn.titleLabel!.text ?? ""
        let rec: Bool = btnTitle.contains("Reset")
        if (rec == true) {
            asReaderGUNManager.mReader?.startHWTriggerDecode()
            btn.setTitle("Stop Scan", for: UIControl.State.normal)
        } else {
            asReaderGUNManager.mReader?.stopHWTriggerDecode()
            btn.setTitle("Reset Scan", for: UIControl.State.normal)
        }
    }
    @IBAction private func rescanTimeBtnTapped(_ sender: UIButton) {
        pickerView.isHidden = false
        pickerView.selectRow(rescanTimeValue, inComponent: 0, animated: false)
        rescanTimePickerTopView.isHidden = false
    }
    @IBAction private func pickerCloseBtnTapped(_ sender: UIButton) {
        hidePickerView()
    }
    @IBAction private func rescanTimePickerOkBtnTapped(_ sender: UIButton) {
        let idx: Int = pickerView.selectedRow(inComponent: 0)
        rescanTimeTextField.text = rescanTimePickerData[idx]
        rescanTimeValue = idx
        systemSetting.setBarcodeReScanTime(Int(rescanTimeValue))
        hidePickerView()
    }
    @IBAction private func outputCSVBtnTapped(_ sender: UIBarButtonItem) {
        let alertController = UIAlertController(title: nil, message: nil, preferredStyle: .actionSheet)
        let cancelAction = UIAlertAction(title: cancel, style: .cancel, handler: nil)
        let csvAction = UIAlertAction(title: "CSV Export", style: .default, handler: { _ in
            self.csvExport()
        })
        let excelAction = UIAlertAction(title: "Excel Export", style: .default, handler: { _ in
            self.excelExport()
        })
        let excelListAction = UIAlertAction(title: "Excel List", style: .default, handler: { _ in
            self.stopScan()
            self.exportType = 2
            self.performSegue(withIdentifier: "barcodeExport", sender: nil)
        })
        alertController.addAction(cancelAction)
        alertController.addAction(csvAction)
        alertController.addAction(excelAction)
        alertController.addAction(excelListAction)
        present(alertController, animated: true, completion: nil)
    }
}
// MARK: - Custom
extension BarcodeDemoViewController {
    private func hidePickerView() {
        pickerView.isHidden = true
        rescanTimePickerTopView.isHidden = true
    }
    private func setScanTimerEnable(_ isEnable: Bool) {
        isScanTimerEnable = isEnable;
        if (!isEnable) {
            NSObject.cancelPreviousPerformRequests(withTarget: self, selector: #selector(startDecoder), object: nil)
        }
    }
    private func scanEnabled(_ enabled: Bool) {
        if enabled {
            asReaderGUNManager.mReader?.startDecode()
            scanElemEnabled(false)
        } else {
            NSObject.cancelPreviousPerformRequests(withTarget: self, selector: #selector(startDecoder), object: nil)
            asReaderGUNManager.mReader?.stopSync()
            scanElemEnabled(true)
        }
    }
    private func scanElemEnabled(_ enabled: Bool) {
        if enabled {
            for control in toDisableElems {
                control.isEnabled = true
            }
            outputBtn.isEnabled = true
        } else {
            for control in toDisableElems {
                control.isEnabled = false
            }
            outputBtn.isEnabled = true
        }
    }
    private func csvExport() {
        DispatchQueue.main.async {
            var rfidCount: Int = 0
            for index in 0..<self.barcodeData.count {
                let item: AsBarcodeItem? = self.barcodeData[index]
                if item?.barcode != nil {
                    rfidCount += 1
                }
            }
            let timeString: String = self.currentData()
            let fileName: String = String.init(format: "barcode_%@.csv", timeString)
            if rfidCount != 0 {
                if CSVData.isFileExist(fileName) {
                    CSVData.deleteFile(forFileName: fileName)
                }
                CSVData.createCSV(withFileName: fileName)
                CSVData.exportBarcodeCSV(fileName, withArray: self.barcodeData)
            } else {
                return
            }
        }
    }
    private func excelExport() {
        DispatchQueue.main.async {
            if (self.barcodeData.count != 0) {
                let timeString: String = self.currentData()
                let fileName: String = String.init(format: "barcode_%@.xlsx", timeString)

                let array: NSMutableArray = NSMutableArray.init()
                for item in self.barcodeData {
                    let fruits = ["barcode":item.barcode, "time":item.dateTime] as [String : Any]
                    array.add(fruits)
                }
                let arr: Array = Array(array)

                ExcelData.createExcel(withFileName: fileName, export: barcodeExportType)
                ExcelData.exportBarcodeExcel(fileName, with: arr, export: barcodeExportType)
            }
        }
    }
    private func currentData() -> String {
        let today: Date = Date.init()
        let df: DateFormatter = DateFormatter.init()
        df.dateFormat = "yyyyMMddHHmmss"
        let str: String = df.string(from: today)
        return str;
    }
    private func stopScan() {
        NSObject.cancelPreviousPerformRequests(withTarget: self, selector: #selector(startDecoder), object: nil)
        setScanTimerEnable(false)
        scanEnabled(false)
    }
    @objc private func startDecoder() {
        scanEnabled(true)
    }
}
// MARK: - UITableViewDelegate, UITableViewDataSource
extension BarcodeDemoViewController: UITableViewDelegate, UITableViewDataSource {
    func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return barcodeData.count
    }
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let cell = tableView.dequeueReusableCell(withIdentifier: "BarcodeCell",
                                                 for: indexPath) as? BarcodeDemoTableViewCell
        let item = barcodeData[indexPath.row]
        cell?.setCodeTypeValue(AsBarcodeType.getString(BarcodeType(rawValue: item.barcodeType)))
        cell?.setCodeIdValue(item.codeId)
        cell?.setBarcodeValue(item.barcode)
        cell?.setCountValue(item.count)
        guard let aCell = cell else {
            return UITableViewCell()
        }
        return aCell
    }
    func tableView(_ tableView: UITableView, editingStyleForRowAt indexPath: IndexPath) -> UITableViewCell.EditingStyle {
        return .delete
    }
}
// MARK: - UIPickerViewDelegate,UIPickerViewDataSource
extension BarcodeDemoViewController: UIPickerViewDelegate, UIPickerViewDataSource {
    func numberOfComponents(in pickerView: UIPickerView) -> Int {
        return 1
    }
    func pickerView(_ pickerView: UIPickerView, numberOfRowsInComponent component: Int) -> Int {
        return rescanTimePickerData.count
    }
    func pickerView(_ pickerView: UIPickerView, titleForRow row: Int, forComponent component: Int) -> String? {
        return rescanTimePickerData[row]
    }
}
// MARK: - AsReaderGUNManagerDelegate
extension BarcodeDemoViewController: AsReaderGUNManagerDelegate {
    func whenAsReaderGUNConnected(_ isConnected: Bool) {
        if isConnected == false {
            DispatchQueue.main.async {
                self.navigationController?.popToRootViewController(animated: true)
            }
        }
    }
    func asReaderGUNDetect(_ barcodeType: BarcodeType, codeId: String!, barcodeData: Data!) {
        //        if barcodeType != BarcodeTypeNoRead {
        hwTriggerBtn.setTitle("Reset Scan", for: .normal)
      
        var barcode: String = String.init(data: barcodeData, encoding: encoding) ?? ""
        if barcode == ""{
            barcode = String.init(data: barcodeData, encoding: .ascii) ?? ""
        }
        var i: Int = 0
        var isNew: Bool = true
        for item in self.barcodeData {
            if (item.barcode == barcode && item.codeId == codeId)
            {
                isNew = false
                item.count = item.count + 1
                self.barcodeData.insert(item, at: i + 1)
                self.barcodeData.remove(at: i)
                break;
            }
            i = i + 1
        }
        if (isNew == true)
        {
            let timeDate = Date()
            let formatter = DateFormatter()
            formatter.dateStyle = .medium
            formatter.timeStyle = .short
            formatter.dateFormat = "YYYY/MM/dd HH:mm:ss"
            let dateTime = formatter.string(from: timeDate)
            let item = AsBarcodeItem(barcodeType: barcodeType, codeId: codeId, barcode: barcode, dateTime: dateTime, count: 1)
            self.barcodeData.append(item)
        }
        systemSetting.setBarcodeData(self.barcodeData)
        DispatchQueue.main.async {
            self.tableView.reloadData()
            self.tableView.scrollToRow(at: IndexPath(row: self.barcodeData.count - 1, section: 0), at: .bottom, animated: true)
            self.totalCountLabel.text = "\(self.barcodeData.count)"
        }
        //        }
    }
    func asReaderGUNChangedActionState(_ action: CommandType, resultCode: Int) {
        if action == CommandStop && isScanTimerEnable {
            if let rescanTime = Double(rescanTimeData[rescanTimeValue]) {
                if rescanTime > 0 {
                    NSObject.cancelPreviousPerformRequests(withTarget: self, selector: #selector(startDecoder), object: nil)
                    perform(#selector(startDecoder), with: nil, afterDelay: rescanTime / 1000.00)
                } else {
                    scanElemEnabled(true)
                }
            }
        }
    }
    func asReaderGUNUpdateDeviceState(_ error: ResultCode) {
        if error != ResultNoError {
            DispatchQueue.main.async {
                let alertController = UIAlertController(title: "Error",
                                                        message: "Failed to get response. Try again.",
                                                        preferredStyle: .alert)
                let okAction = UIAlertAction(title: ok, style: .default, handler: { _ in
                    self.navigationController?.popViewController(animated: true)
                })
                alertController.addAction(okAction)
                self.present(alertController, animated: true, completion: nil)
            }
        }
    }
    func asReaderGUNOn(asReaderTriggerKeyEvent status: Bool) -> Bool {
        setScanTimerEnable(status)
        if (status){
            scanEnabled(true)
        } else {
            scanEnabled(false)
        }
        return false;
    }
    func asReaderGUNOn(asReaderLeftModeKeyEvent status: Bool) -> Bool {
        let rfidModule: Bool = (asReaderGUNManager.mReader?.isRFIDModule())!
        let barcodeModule: Bool = (asReaderGUNManager.mReader?.isBarcodeModule())!
        if  rfidModule && barcodeModule {
            return true
        } else {
            return false
        }
    }
    func asReaderGUNOn(asReaderRightModeKeyEvent status: Bool) -> Bool {
        let rfidModule: Bool = (asReaderGUNManager.mReader?.isRFIDModule())!
        let barcodeModule: Bool = (asReaderGUNManager.mReader?.isBarcodeModule())!
        if  rfidModule && barcodeModule {
            return true
        } else {
            return false
        }
    }
}
