//
//  BarcodeOptionVC.swift
//  AsReaderGUN
//
//  Created by mac on 2018/7/12.
//  Copyright © 2018年 asterisk. All rights reserved.
//

import UIKit

final class BarcodeOptionVC: UIViewController {
    @IBOutlet private weak var scrollView: UIScrollView!
    @IBOutlet private weak var pickerView: UIPickerView!
    @IBOutlet private weak var buzzerTextField: UITextField!
    @IBOutlet private weak var vibratorTextField: UITextField!
    @IBOutlet private weak var batteryIntervalTextField: UITextField!
    @IBOutlet private weak var buzzerPickerTopView: UIView!
    @IBOutlet private weak var vibratorPickerTopView: UIView!
    @IBOutlet private weak var shadowView: UIView!
    @IBOutlet private weak var sleepTimeTextField: UITextField!
    private enum PickerViewTag: Int {
        case pickerViewBuzzer = 0
        case pickerViewVibrator = 1
    }
    private var buzzerValue: BuzzerState?
    private var vibratorValue: VibratorState?
    private var batteryIntervalValue = ""
    private var sleepTimeValue = 0
    private var asReaderGUNManager: AsReaderGUNManager! = AsReaderGUNManager.sharedAsReaderGUNManager
    private var systemSetting: SystemSetting! = SystemSetting.shared()
    private lazy var buzzerPickerData: [String] = {
        return dataBuzzerPicker
    }()
    private lazy var vibratoryPickerData: [String] = {
        return dataVibratoryPicker
    }()
    override func viewDidLoad() {
        super.viewDidLoad()
        scrollView.contentSize = CGSize(width: 0, height: 1000)
        scrollView.showsVerticalScrollIndicator = false
    }
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        asReaderGUNManager.asReaderGUNManagerDelegate = self
    }
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(true)
        AsReaderGUNManager.showGlobalProgressHUD(withTitle: loading)
        if asReaderGUNManager.mReader == nil {
            AsReaderGUNManager.dismissGlobalHUD()
            return
        }
        vibratorValue = asReaderGUNManager.mReader?.vibrator
        buzzerValue = asReaderGUNManager.mReader?.buzzer
        if let sleepTime = asReaderGUNManager.mReader?.sleepTime {
            sleepTimeValue = Int(sleepTime)
        }
        initViewData()
        navigationController?.view.addSubview(shadowView)
        AsReaderGUNManager.dismissGlobalHUD()
    }
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        
    }
    override func didReceiveMemoryWarning() {
        super.didReceiveMemoryWarning()
    }
    @IBAction private func buzzerBtnTapped(_ sender: UIButton) {
        pickerView.tag = PickerViewTag.pickerViewBuzzer.rawValue
        pickerView.reloadAllComponents()
        buzzerPickerTopView.isHidden = false
        showPickerView()
        if let text = buzzerTextField.text {
            if !text.isEmpty {
                if let row = buzzerPickerData.firstIndex(of: text) {
                    pickerView.selectRow(row, inComponent: 0, animated: false)
                    return
                }
            }
        }
        pickerView.selectRow(0, inComponent: 0, animated: false)
    }
    @IBAction private func vibratorBtnTapped(_ sender: UIButton) {
        pickerView.tag = PickerViewTag.pickerViewVibrator.rawValue
        pickerView.reloadAllComponents()
        vibratorPickerTopView.isHidden = false
        showPickerView()
        if let text =  vibratorTextField.text {
            if !text.isEmpty {
                if let row = vibratoryPickerData.firstIndex(of: text) {
                    pickerView.selectRow(row, inComponent: 0, animated: false)
                    return
                }
            }
        }
        pickerView.selectRow(0, inComponent: 0, animated: false)
    }
    @IBAction private func batteryIntervalBtnTapped(_ sender: UIButton) {
        let alertController = UIAlertController(title: "Battery Interval:", message: "Please enter battery interval.", preferredStyle: .alert)
        alertController.addTextField { (textField) in
            textField.keyboardType = .numberPad
            textField.delegate = self
        }
        let cancelAction = UIAlertAction(title: cancel, style: .cancel, handler: nil)
        let okAction = UIAlertAction(title: ok, style: .default, handler: { _ in
            var text = alertController.textFields?.first?.text
            guard let value = text else {
                return
            }
            if value.isEmpty {
                text = "0"
            }
            self.setBatteryInterval(text)
        })
        alertController.addAction(cancelAction)
        alertController.addAction(okAction)
        present(alertController, animated: true, completion: nil)
    }
    @IBAction private func sleepTimeBtnTapped(_ sender: UIButton) {
        let alertController = UIAlertController(title: "sleep Time:", message: "Please enter sleep time.", preferredStyle: .alert)
        alertController.addTextField { (textField) in
            textField.keyboardType = .numberPad
            textField.delegate = self
        }
        let cancelAction = UIAlertAction(title: cancel, style: .cancel, handler: nil)
        let okAction = UIAlertAction(title: ok, style: .default, handler: { _ in
            var text = alertController.textFields?.first?.text
            guard let value = text else {
                return
            }
            if value.isEmpty {
                text = "0"
            }
            self.setSleepTime(text)
        })
        alertController.addAction(cancelAction)
        alertController.addAction(okAction)
        present(alertController, animated: true, completion: nil)
    }
    @IBAction private func saveBtnTapped(_ sender: UIButton) {
        if let aBuzzerValue = buzzerValue {
            asReaderGUNManager.mReader?.buzzer = aBuzzerValue
        }
        if let aVibratorValue = vibratorValue {
            asReaderGUNManager.mReader?.vibrator = aVibratorValue
        }
        asReaderGUNManager.mReader?.sleepTime = Int32(sleepTimeValue)
        if let buzzerValue = buzzerValue?.rawValue {
            systemSetting.setBuzzer(Int(buzzerValue))
        }
        if let vibratorValue = vibratorValue?.rawValue {
            systemSetting.setVibrator(Int(vibratorValue))
        }
        systemSetting.setSleepTime(Int(sleepTimeValue))
        systemSetting.setBatteryInterval(batteryIntervalValue)
        let alertController = UIAlertController(title: "Success",
                                                message: "Saved successfully",
                                                preferredStyle: .alert)
        let okAction = UIAlertAction(title: ok, style: .default, handler: nil)
        alertController.addAction(okAction)
        present(alertController, animated: true, completion: nil)
    }
    @IBAction private func defaultBtnTapped(_ sender: UIButton) {
        asReaderGUNManager.mReader?.buzzer = Buzzer_High
        asReaderGUNManager.mReader?.vibrator = Vibrator_On
        asReaderGUNManager.mReader?.sleepTime = 300
        buzzerValue = asReaderGUNManager.mReader?.buzzer
        vibratorValue = asReaderGUNManager.mReader?.vibrator
        systemSetting.setSleepTime(Int(sleepTimeValue))
        if let sleepTime = asReaderGUNManager.mReader?.sleepTime {
            sleepTimeValue = Int(sleepTime)
        }
        if let aBuzzerValue = buzzerValue?.rawValue {
            systemSetting.setBuzzer(Int(aBuzzerValue))
        }
        if let aVibratorValue = vibratorValue?.rawValue {
            systemSetting.setVibrator(Int(aVibratorValue))
        }
        systemSetting.setBatteryInterval("10000")
        initViewData()
        let alertController = UIAlertController(title: "Success",
                                                message: "Set to default successfully",
                                                preferredStyle: .alert)
        let okAction = UIAlertAction(title: ok, style: .default, handler: nil)
        alertController.addAction(okAction)
        present(alertController, animated: true, completion: nil)
    }
    @IBAction private func pickerCloseBtnTapped(_ sender: UIButton) {
        hidePickerView()
    }
    @IBAction private func buzzerPickerOkBtnTapped(_ sender: UIButton) {
        let idx: Int = pickerView.selectedRow(inComponent: 0)
        setBuzzer(idx)
        hidePickerView()
    }
    @IBAction private func vibratorPickerOkBtnTapped(_ sender: UIButton) {
        let idx: Int = pickerView.selectedRow(inComponent: 0)
        setVibrator(idx)
        hidePickerView()
    }
}
// MARK: - Custom
extension BarcodeOptionVC {
    func showPickerView() {
        pickerView.isHidden = false
        shadowView.isHidden = false
    }
    func hidePickerView() {
        vibratorPickerTopView.isHidden = true
        buzzerPickerTopView.isHidden = true
        pickerView.isHidden = true
        shadowView.isHidden = true
    }
    func hideTableView() {
        shadowView.isHidden = true
    }
    func setBuzzer(_ index: Int) {
        buzzerTextField.text = buzzerPickerData[index]
        buzzerValue = BuzzerState(rawValue: BuzzerState.RawValue(index))
    }
    func setVibrator(_ index: Int) {
        vibratorTextField.text = vibratoryPickerData[index]
        vibratorValue = VibratorState(rawValue: VibratorState.RawValue(index))
    }
    func setBatteryInterval(_ batteryInterval: String!) {
        batteryIntervalTextField.text = batteryInterval
        batteryIntervalValue = batteryInterval
    }
    func setSleepTime(_ sleepTime: String!) {
        sleepTimeTextField.text = sleepTime
        sleepTimeValue = Int(sleepTime) ?? 0
    }
    func initViewData() {
        if let buzzerValue =  buzzerValue?.rawValue {
            setBuzzer(Int(buzzerValue))
        }
        if let vibratorValue =  vibratorValue?.rawValue {
            setVibrator(Int(vibratorValue))
        }
        setSleepTime("\(sleepTimeValue)")
        let batteryInterval = systemSetting.getBatteryInterval()
        setBatteryInterval(batteryInterval)
    }
}
// MARK: - UIPickerViewDelegate, UIPickerViewDataSource
extension BarcodeOptionVC: UIPickerViewDelegate, UIPickerViewDataSource {
    func numberOfComponents(in pickerView: UIPickerView) -> Int {
        return 1
    }
    func pickerView(_ pickerView: UIPickerView, numberOfRowsInComponent component: Int) -> Int {
        if let pickerViewTag = PickerViewTag(rawValue: pickerView.tag) {
            if pickerViewTag == .pickerViewBuzzer {
                return buzzerPickerData.count
            } else if pickerViewTag == .pickerViewVibrator {
                return vibratoryPickerData.count
            }
        }
        return 0
    }
    func pickerView(_ pickerView: UIPickerView, titleForRow row: Int, forComponent component: Int) -> String? {
        if let pickerViewTag = PickerViewTag(rawValue: pickerView.tag) {
            if pickerViewTag == .pickerViewBuzzer {
                return buzzerPickerData[row]
            } else if pickerViewTag == .pickerViewVibrator {
                return vibratoryPickerData[row]
            }
        }
        return "None"
    }
}
// MARK: - AsReaderGUNManagerDelegate
extension BarcodeOptionVC: AsReaderGUNManagerDelegate {
    func whenAsReaderGUNConnected(_ isConnected: Bool) {
        if isConnected == false {
            DispatchQueue.main.async {
                self.navigationController?.popToRootViewController(animated: true)
            }
        }
    }
    func asReaderGUNOn(asReaderTriggerKeyEvent status: Bool) -> Bool {
        return false
    }
    func asReaderGUNOn(asReaderLeftModeKeyEvent status: Bool) -> Bool {
        let rfidModule: Bool = (asReaderGUNManager.mReader?.isRFIDModule())!
        let barcodeModule: Bool = (asReaderGUNManager.mReader?.isBarcodeModule())!
        if  rfidModule && barcodeModule {
            return true
        } else {
            return false
        }
    }
    func asReaderGUNOn(asReaderRightModeKeyEvent status: Bool) -> Bool {
        let rfidModule: Bool = (asReaderGUNManager.mReader?.isRFIDModule())!
        let barcodeModule: Bool = (asReaderGUNManager.mReader?.isBarcodeModule())!
        if  rfidModule && barcodeModule {
            return true
        } else {
            return false
        }
    }
    func asReaderGUNUpdateDeviceState(_ error: ResultCode) {
        if error != ResultNoError {
            DispatchQueue.main.async {
                let alertController = UIAlertController(title: "Error",
                                                        message: "Failed to get response. Try again.",
                                                        preferredStyle: .alert)
                let okAction = UIAlertAction(title: ok, style: .default, handler: { _ in
                    self.navigationController?.popViewController(animated: true)
                })
                alertController.addAction(okAction)
                self.present(alertController, animated: true, completion: nil)
            }
        }
    }
}
// MARK: - UITextFieldDelegate
extension BarcodeOptionVC: UITextFieldDelegate {
    func textField(_ textField: UITextField, shouldChangeCharactersIn range: NSRange, replacementString string: String) -> Bool {
        let text = textField.text ?? ""
        let length = text.count + string.count - range.length
        return length <= 9
    }
}
