package jp.co.asterisk.asreader.a3xd.demoapp.fragment.tagaccess;

import android.app.AlertDialog;
import android.app.Dialog;
import android.content.DialogInterface;
import android.os.Bundle;
import android.text.Editable;
import android.text.InputFilter;
import android.text.TextWatcher;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.RadioGroup;
import android.widget.Toast;

import jp.co.asterisk.asreader.a3xd.demoapp.databinding.FragmentReadWriteBinding;
import jp.co.asterisk.asreader.a3xd.demoapp.fragment.BaseFragment;
import jp.co.asterisk.asreader.a3xd.sdk.AsReaderSDK;
import jp.co.asterisk.asreader.a3xd.sdk.common.Convert;
import jp.co.asterisk.asreader.a3xd.sdk.common.RFIDConst;
import jp.co.asterisk.asreader.a3xd.sdk.manager.RFIDManagerCallback;
import jp.co.asterisk.asreader.a3xd.sdk.sdevice.AsDeviceLib;
import jp.co.asterisk.asreader.a3xd.demoapp.R;

public class ReadWriteFragment extends BaseFragment implements RFIDManagerCallback {

    private final String errMessageEPCField = "An invalid value was entered in EPC field";
    private final String errMessageDataLengthField = "An invalid value was entered in DataLength field";
    private final String errMessagePasswordField = "An invalid value was entered in Password field";
    private final String getErrMessageStartAddress = "An invalid value was entered in Start Address field";

    enum ErrorMessage {
        INVALID_EPC,
        INVALID_DATA_LENGTH,
        INVALID_PASSWORD,
        INVALID_START_ADDRESS;
    };

    private final String TAG = ReadWriteFragment.class.getSimpleName();

    public static final String ARGS_SELECTED_TAG = "selectTag";

    private FragmentReadWriteBinding binding;

    private boolean state = true;
    private RFIDConst.MemoryBank memoryBank = RFIDConst.MemoryBank.RESERVED;
    private int dataAddr;
    private String nowTag;
    private String dataText;
    private String addressText;
    private StringBuffer dataBuffer;
    private String title;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        if (getArguments() != null) {
            nowTag = getArguments().getString(ARGS_SELECTED_TAG);
        }
        title = getString(R.string.toolbar_title_read);
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        setBackToHomeActionResId(R.id.action_ReadWriteFragment_to_HomeFragment);

        binding = FragmentReadWriteBinding.inflate(inflater, container, false);

        binding.targetTagEditText.setText(nowTag);
        if (memoryBank== RFIDConst.MemoryBank.RESERVED) binding.wordLengthEditText.setText("4");
        binding.readWriteRadioGroup.setOnCheckedChangeListener(onCheckedChangeListener);
        binding.targetMemoryRadioGroup.setOnCheckedChangeListener(onCheckedChangeListener);
        binding.startAddressEditText.addTextChangedListener(textWatcher);
        return binding.getRoot();
    }

    @Override
    public void onResume() {
        super.onResume();
        AsReaderSDK.getInstance().getRFIDManager().setCallback(this);

        setToolbar(true, title, getString(R.string.button_name_done));
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        binding = null;
    }

    @Override
    public void onClickRightButton(View view) {
        super.onClickRightButton(view);
        if (state) {
            readMemory();
        } else {
            writeMemory();
        }
    }

    private void showAlertDialog( String message ) {
        AlertDialog.Builder builder = new AlertDialog.Builder(this.getActivity());
        builder.setTitle("Alert!!");
        builder.setMessage(message);

        builder.setPositiveButton("OK", new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialog, int id) {
                // User clicked OK button
                dialog.dismiss();
            }
        });

        AlertDialog dialog = builder.create();
        dialog.show();
    }

    private String getErrorMessage(ErrorMessage type){
        switch (type){
            case INVALID_EPC:
                return errMessageEPCField;
            case INVALID_DATA_LENGTH:
                return errMessageDataLengthField;
            case INVALID_PASSWORD:
                return errMessagePasswordField;
            case INVALID_START_ADDRESS:
                return getErrMessageStartAddress;
            default:
                return "UnKnown";
        }
    }

    private boolean checkParameterAndShowDialog(String dataStr, String errMessage) {

        // EPC
        if( dataStr==null || dataStr.length()==0 || dataStr=="") {
            showAlertDialog(errMessage);
            return false;
        }

        return true;
    }

    private void readMemory() {
        binding.dataEditText.setText("");

        String epc = binding.targetTagEditText.getText().toString();
        String dataLengthStr = binding.wordLengthEditText.getText().toString();
        String passwordStr = binding.accessPasswordEditText.getText().toString();
        String startAddressStr = binding.startAddressEditText.getText().toString();

        if( checkParameterAndShowDialog(epc, getErrorMessage(ErrorMessage.INVALID_EPC))==false) {
            return;
        } else if( checkParameterAndShowDialog(dataLengthStr, getErrorMessage(ErrorMessage.INVALID_DATA_LENGTH))==false) {
            return;
        } else if( checkParameterAndShowDialog(passwordStr, getErrorMessage(ErrorMessage.INVALID_PASSWORD))==false) {
            return;
        } else if( checkParameterAndShowDialog(startAddressStr, getErrorMessage(ErrorMessage.INVALID_START_ADDRESS))==false) {
            return;
        }

        try {
            int dataLengthValue = Integer.parseInt(dataLengthStr);
            RFIDConst.ReadMemoryType readType = RFIDConst.ReadMemoryType.NORMAL;
            
            if (dataLengthValue > 64) {
                // Sets the maxLength of "tvData"
                readType = RFIDConst.ReadMemoryType.LONG_DATA;
                InputFilter[] FilterArray = new InputFilter[1];
                FilterArray[0] = new InputFilter.LengthFilter((dataLengthValue >> 2) * 40);
                binding.dataEditText.setFilters(FilterArray);

                dataAddr = 0;
                dataBuffer = new StringBuffer();
            }

            AsReaderSDK.getInstance().getRFIDManager().readTagMemory(
                    Long.parseLong(passwordStr, 16),
                    epc,
                    memoryBank,
                    readType,
                    Integer.parseInt(startAddressStr),
                    dataLengthValue,
                    0);

        } catch (Exception e) {
            e.printStackTrace();
            showToast(getString(R.string.error));
        }
    }

    private void writeMemory() {
        String epc = binding.targetTagEditText.getText().toString();
        String dataLengthStr = binding.wordLengthEditText.getText().toString();
        String passwordStr = binding.accessPasswordEditText.getText().toString();
        String startAddressStr = binding.startAddressEditText.getText().toString();

        String data_temp = binding.dataEditText.getText().toString();

        if( checkParameterAndShowDialog(epc, getErrorMessage(ErrorMessage.INVALID_EPC))==false) {
            return;
        } else if( checkParameterAndShowDialog(dataLengthStr, getErrorMessage(ErrorMessage.INVALID_DATA_LENGTH))==false) {
            return;
        } else if( checkParameterAndShowDialog(passwordStr, getErrorMessage(ErrorMessage.INVALID_PASSWORD))==false) {
            return;
        } else if( checkParameterAndShowDialog(startAddressStr, getErrorMessage(ErrorMessage.INVALID_START_ADDRESS))==false) {
            return;
        }

        if(data_temp.length() % 4 != 0) {
            showToast("Data length is invalid.");
        } else {
            try {
                AsReaderSDK.getInstance().getRFIDManager().writeTagMemory(
                        Long.parseLong(passwordStr, 16),
                        epc,
                        memoryBank,
                        RFIDConst.WriteMemoryType.NORMAL,
                        Integer.parseInt(startAddressStr),
                        Convert.toBytes(data_temp),
                        0 );

                binding.wordLengthEditText.setText(String.valueOf(data_temp.length() / 4));

            } catch (Exception e) {
                e.printStackTrace();
                showToast(getString(R.string.error));
            }
        }
    }

    private RadioGroup.OnCheckedChangeListener onCheckedChangeListener = new RadioGroup.OnCheckedChangeListener() {
        @Override
        public void onCheckedChanged(RadioGroup group, int checkedId) {
            if (group == binding.readWriteRadioGroup) {
                if (checkedId == R.id.read_radio_button) {
                    state = true;
                    title = getString(R.string.toolbar_title_read);
                    setTitleText(title);
                    binding.wordLengthEditText.setEnabled(true);

                } else if (checkedId == R.id.write_radio_button) {
                    state = false;
                    title = getString(R.string.toolbar_title_write);
                    setTitleText(title);
                    binding.wordLengthEditText.setEnabled(false);

                }

            } else if (group == binding.targetMemoryRadioGroup) {
                if (checkedId == R.id.rfu_radio_button) {
                    memoryBank = RFIDConst.MemoryBank.RESERVED;

                } else if (checkedId == R.id.epc_radio_button) {
                    memoryBank = RFIDConst.MemoryBank.EPC;

                } else if (checkedId == R.id.tid_radio_button) {
                    memoryBank = RFIDConst.MemoryBank.TID;

                } else if (checkedId == R.id.user_radio_button) {
                    memoryBank = RFIDConst.MemoryBank.USER;
                }
            }

            if (memoryBank == RFIDConst.MemoryBank.EPC) {
                binding.startAddressEditText.setText("2");
            }else {
                binding.startAddressEditText.setText("0000");
            }

            if (memoryBank == RFIDConst.MemoryBank.RESERVED) {
                binding.wordLengthEditText.setText("4");
            }else {
                binding.wordLengthEditText.setText("0");
            }
        }
    };

    private TextWatcher textWatcher = new TextWatcher() {
        @Override
        public void beforeTextChanged(CharSequence s, int start, int count, int after) {

        }

        @Override
        public void onTextChanged(CharSequence s, int start, int before, int count) {
            addressText = s.toString();
        }

        @Override
        public void afterTextChanged(Editable s) {

        }
    };

    public String editAddressString(int dataAddr, StringBuffer dataString) {
        StringBuffer sb = new StringBuffer();

        for(int i = 0; i < dataString.length(); i++) {
            if(i % 16 == 0) {
                sb.append(String.format("0x%04X    ", dataAddr + (i / 4)));
            }
            sb.append(dataString.charAt(i));

            if(i % 2 == 1) {
                sb.append(" ");
            }

            if(i % 16 == 15) {
                sb.append("\n");
            }
        }

        return sb.toString();
    }

    @Override
    public void onWriteToTagMemorySuccess() {
        if (getActivity()!=null){
            getActivity().runOnUiThread(() -> {
                showToast(getString(R.string.success), Toast.LENGTH_SHORT);
            });
        }
    }

    @Override
    public void onTagMemoryReceived(int[] data) {
        dataText = Convert.toStringHEX(data);

        if (getActivity()!=null){
            getActivity().runOnUiThread(() -> {
                binding.dataEditText.setText(dataText);
            });
        }
    }

    @Override
    public void onTagMemoryLongReceived(int[] dest) {
        if (dest.length > 3) {
            dataAddr = Integer.parseInt(addressText, 16);
            dataBuffer.append(Convert.toStringHEX(dest).substring(6));

            if (getActivity()!=null){
                getActivity().runOnUiThread(() -> {
                    binding.dataEditText.setText("Read operation is in progress..");
                });
            }
        } else if((dest.length == 1) & (dest[0] == 0x1F)) { // 0x1F = Complete Read Memory
            if (getActivity()!=null){
                getActivity().runOnUiThread(() -> {
                    binding.dataEditText.setText(editAddressString(dataAddr, dataBuffer));
                });
            }
        }
    }

    @Override
    public void onRFIDManagerError(String message) {
        Log.d(TAG, message);
    }
}
